/**
 * Copyright (C) 2025. Donald Pickett All rights reserved
 * Provided as part the Gauge 1 3D Circle
 * License: Creative Common Attribution-NonCommercial-ShareAlike CC BY-NC-SA 4.0 International
 *          See: https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode.en
 * Version 1.0 03/01/2025
 */

module roundedCylinder(h,d,r,r2=360,center=true) {
  da=90/($fn<12?12:$fn);
  assert(r<=d/2,"capsue: Unworkable parameters- r must be <= d/2");

  r0=d/2;
  points=
  [
    [0,0],
    for (a=[0:da:90]) [r0-r+r*sin(a),(r*(1-cos(a)))],
    for (a=[0:da:90]) [r0-(r*(1-cos(a))),h-r*(1-sin(a))],
    [0,h]
  ];
  translate([0,0,(center?-h/2:0)])
    rotate_extrude(angle=r2)
      polygon(points);  
}

/* Creates a fillet using the radius that gives one size with a thickness of t
 */
module fillet(size,t=0,center=true) {
  x=size[0];
  y=size[1];
  z=size[2];

  r=min(y-t,z-t);
  assert(t>=0,"fillet: - Unworkable parameters t must be >= 0");
  assert(r>0,"fillet: - Unworkable parameters t must be < y and z");
  
  da=180/($fn<12?12:$fn);
  points=[
    [-y/2+r,-z/2],
    for (a=[0:da:90]) [-y/2+(r*sin(a)),-z/2+(r*cos(a))],
    [-y/2,-z/2+r],
    [-y/2,z/2],
    [y/2,z/2],
    [y/2,-z/2],
  
  ];

  linear_extrude(x,center=center)
    polygon(points);  
}

/** Creates a doughnut shape (or part of)
 * d: diameter of center of doughnut
 * t: thickness of doughnut
 * a: (optional) andgle of part of doughnut (360 by default)
 */
module doughnut(d,t,a=360) {
  rotate_extrude(angle=a)
    translate([d/2,0])
      circle(d=t);
}

/* Used to create an eliptical sided cone of height h and diameter d1 and d2.
 */
module elipticHorn(h,d1,d2,a=360) {
  da=180/($fn<12?12:$fn);
  r1=d1/2;
  r2=d2/2;
  points= [
    [d2/2,0],  
    [r2,h],
    [0,h],
    [0,0],
    [r1,0],
    for (ax=[0:da:90]) [r2+((r1-r2)*(1-sin(ax))),h*(1-cos(ax))],
  ];
  
  rotate_extrude(angle=a)
    polygon(points);  
}

/** Draws a 2D oval
 * w = Width - length of major axis
 * h = Height - length of minor axis
 * r = radius of minor circle (on ends of major axis
 * Note j = distance from center axis to centre of larger arc
 *      s = Angle of arc of minor circle from centre axis
 */
module oval(w,h,r) {
  assert(r<h/2,"oval: Unworkable parameters- r must be < h/2");
  assert(w>h, "oval: Unworkable parameters- w must be > h"); 
  
  a = w/2;
  b = h/2;
  k= a-r;
  
  j = (((a^2 - b^2 - 2*a*r)/2) + r*b)/(b-r);
  r2=j+b;  
  s=atan(j/k);
  t=90-s;  
  da=180/($fn<80?80:$fn);
  p = [
   [k+(r*cos(-s)),(r*sin(-s))],
   for (x=[-s:da:s]) [k+(r*cos(x)),(r*sin(x))],
   [k+(r*cos(s)),(r*sin(s))],
   [(r2*sin(t)),(r2*cos(t))-j],
   for (x=[t:-da:-t]) [(r2*sin(x)),(r2*cos(x))-j],
   [(r2*sin(-t)),(r2*cos(-t))-j],
   [-k-(r*cos(s)),(r*sin(s))],
   for (x=[s:-da:-s]) [-k-(r*cos(x)),(r*sin(x))],
   [-k-(r*cos(-s)),(r*sin(-s))],
   [(r2*sin(-t)),-(r2*cos(-t))+j],
   for (x=[-t:da:t]) [(r2*sin(x)),-(r2*cos(x))+j],
   [(r2*sin(t)),-(r2*cos(t))+j],
  ];
  
   polygon(p);
}

/* Provides various screw heads.
 * Parameters:
 *  h = Height of screw head
 *  d = Diameter of screw
 *  type = Type of screw
 *    "ch"  = cheese head with slot
 *    "ph"  = pan head with slot
 *    "cb"  = coach bolt (flatted dome) 
 *    "pan" = pan head with no slot
 *    "nut" = nut on cylinder 
 *    "nut&bolt" = nut on cylinder together with a bolt head
 *    "snut"= square nut on cylinder 
 *    "pin" = Pin through hold with washer and split pin
 *    "rivet" = rivet head (no washer)
 *    "csk" = countersunk screw (See also use of "holes" variable in fixingArray() module
 *    "bolt" = Bolt head (no washer)
 *    default is a hex bolt head
 *  washer = Add a washer below fixing default TRUE but does not applly to rivet
 *  random = Rotates the object random so that nuts don't all allign
 *  ah     = Additional height (can be used to elongate the bolt within a nut)
 *  wt     = Washer thickness override (default to h*0.2)
 *  anchor = Add anchor into surface (may not be wanted on thin surfaces)
 *  l = Length of nut + bolt (Distance between faces)
 */
module screwHead(h,d,type="ch",washer=true,random=false,ah=0,wt=0,anchor=true,l=0) {
  wat=(washer && type!="rivet" && type!="pin")?(wt==0)?h*0.2:wt:0;
  if (wat>0) {
    cylinder(h=wat,d=d*(type=="bolt"?1.25:1.25));
  }
  if (type=="ch") {     // Cheese head
    rotate([0,0,random?rands(0,60,1)[0]:0])
      difference() {
        cylinder(h=h,d=d);
        translate([0,0,h*3/4+_dx])
          cube([d/6,d,h/2],center=true);
    }
  } else if (type=="ph") {     // Pan head
    rotate([0,0,random?rands(0,60,1)[0]:0])
      difference() {
        halfRoundedCylinder(h=h,d=d,r=d/6);
        translate([0,0,h*3/4+_dx])
          cube([d/6,d,h/2],center=true);
      }
  } else if (type=="cb") {     // Pan head
//    halfRoundedCylinder(h=h,d=d,r=d/6);
    dome(h=h,d=d);
  } else if (type=="pan") {
    halfRoundedCylinder(h=h,d=d,r=d/8);
  } else if (type=="rivet") {
    halfRoundedCylinder(h=max(h,d/2),d=d,r=d/2);
  } else if (type=="csk") {
    rotate([0,0,random?rands(0,360,1)[0]:0])
      difference() {
        cylinder(h=h,d=d);       // Plain cylinder with slot
        translate([0,0,h*3/4+_dx])
          cube([d/6,d,h/2],center=true);
      }
  } else if (type=="nut") {
    cylinder(h=h+ah,d=d*0.6);      // The end of the thread
    rotate([0,0,random?rands(0,60,1)[0]:0])
      cylinder(h=h*0.9,d=d,$fn=6);    // Nut head.
  } else if (type=="bolt") {
    rotate([0,0,random?rands(0,60,1)[0]:0])
      cylinder(h=h*0.9,d=d,$fn=6);    // Nut head.
  } else if (type=="nut&bolt") {
    //translate([0,1,-l]) color("red") sphere(0.1);
    // First a standard nut
    translate([0,0,-l-h*0.8])
      cylinder(h=l+h*1.8+ah,d=d*0.6);      // The end of the thread
    rotate([0,0,random?rands(0,60,1)[0]:0])
      cylinder(h=h*0.9,d=d,$fn=6);    // Nut head.
     
    // Now the bolt 
    translate([0,0,-l-h*0.8-wat])   {
      cylinder(h=h*0.8,d=d,$fn=6);    // Nut head.
      if (wat>0) {
        translate([0,0,h*0.8])
          cylinder(h=wat,d=d*(type=="bolt"?1.25:1.25));
      }
    }
  } else if (type=="snut") {
    cylinder(h=h+ah,d=d*0.5);      // The end of the thread
    rotate([0,0,random?rands(0,60,1)[0]:0])
      translate([0,0,wat+h/4])
        cube([d*0.8,d*0.9,h/2],center=true);    // Nut head.
  } else if (type=="pin") {
    translate([0,0,-h])
      cylinder(h=h,d=d*0.6);                  // The pin shaft
    rotate([0,0,random?rands(0,60,1)[0]:0])
      halfRoundedCylinder(h=d/4,d=d,r=d/6);   // Pin head (pan shaped)
    // Washer near end of pin
    translate([0,0,-h+d*0.15])
      cylinder(h=d/8,d=d*1.25);
    // Now the "split pin" retainer
    st=d/10;
    rotate([0,0,rands(0,360,1)[0]]) {
      translate([0,d*0.4,-h+d/8])
        doughnut(d=d/5,t=st);
      translate([0,-d*0.3,-h+d/8]) {
        rotate([90,0,rands(5,30,1)[0]])
          cylinder(h=d/2,d=st);
        rotate([90,0,-rands(5,50,1)[0]])
          cylinder(h=d/2,d=st);
      }
    }
  } else {
    rotate([0,0,random?rands(0,60,1)[0]:0])
      cylinder(h=h,d=d,$fn=6);    // Bolt head.
  }
  // Add an achor to ensure it sticks to surface
  if (anchor && type !="pin") {
    ed=_embedDepth/2;     // Just a bit!
    translate([0,0,-ed/2+_dx]) cylinder(h=ed,d=d*3/4,center=true);
  }  
}

module halfRoundedCylinder(h,d,r,r2=360,center=false) {
  da=90/($fn<12?12:$fn);
  assert(r<=d/2,"capsue: Unworkable parameters- r must be <= d/2");

  r0=d/2;
  points=
  [
    [0,0],
    [r0,0],
    for (a=[0:da:90]) [r0-(r*(1-cos(a))),h-r*(1-sin(a))],
    [0,h]
  ];
  translate([0,0,center?-h/2:0])
    rotate_extrude(angle=r2)
      polygon(points);  
}

/** Draws a 2D egg shape using two circles joined by tangents.
 * Parameters:
 * r1 = radius of circle located on origin
 * r2 = radius of 2nd circel
 * dx = distance between the centres of the two circles
 * xOff - Offset along x axis
 * yOff - Offset along y axis
 */
module flatEgg(r1,r2,dx,xOff=0,yOff=0) {
  r3=r1-r2;     // Difference in radii
  y = sqrt(dx*dx-r3*r3);  // Centre of outside circle to tangent of origin circle`
  a=acos(y/dx)*((r1>=r2)?1:-1);  // Angle of tangent
  da=180/($fn<12?12:$fn); // Incremental steps
  p=[
    [xOff+r1*sin(a),yOff+r1*cos(a)], 
    for(i=[a:-da:-180-a]) [xOff+r1*(sin(i)),yOff+r1*cos(i)],
    [xOff+r1*sin(a),yOff-r1*cos(a)],
    
    [xOff+dx+r2*sin(a),yOff-r2*cos(a)],
    for(i=[180-a:-da:a]) [xOff+dx+r2*(sin(i)),yOff+r2*cos(i)],
    [xOff+dx+r2*sin(a),yOff+r2*cos(a)],
  ];
 polygon(p);
}

module extrudedFlatEgg(r1,r2,dx,t,center=true) {
  linear_extrude(t,center=center) 
    flatEgg(r1,r2,dx);
}
