/**
 * Copyright (C) 2025. Donald Pickett All rights reserved
 * Provided as part the Gauge 1 3D Circle
 * License: Creative Common Attribution-NonCommercial-ShareAlike CC BY-NC-SA 4.0 International
 *          See: https://creativecommons.org/licenses/by-nc-sa/4.0/legalcode.en
 * Version 1.4 21/10/2025
 *   Bug fixed when scaling chamfers on spoked wheels
 * Version 1.2 09/01/2025
 *  Added split spoke type 
 *  Added ability to alter number and size of holes in wheels
 *  Fixed rims size for plain hole wheels
 *  Minor adjustment to position of wheel hole positions
 * Version 1.1 03/01/2025
 *  Added RCH 3 hole pressed steel wheel
 *  Minor code changes and tidy ups.
 * Version 1.0 03/01/2025
 *  Initial release version
 */

smm=25.4;
sf=1;     // Default value

// The following data is in inches for guage 1 purposes
wheelProfileData =
[
  [
    "g1Standard",       // 
    [
      6.0,      // 0: Tyre width
      3,        // 1: Tyre coning angle
      2.0,      // 2: Flange depth
      1.5,      // 3: Flange width
      20,       // 4: Flange angle
      0.5,      // 5: Root radius
      [         // 6: Flange tip radii
        0.5,      // 
      ],
      0.2,      // 7: Chamfer depth  // Does not seem to be specified
      30,       // 8: Chamfer angle
      40,       // 9: Back to back measurement
      45,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "g1Fine",       // 
    [
      5.0,      // 0: Tyre width
      3,        // 1: Tyre coning angle
      1.5,      // 2: Flange depth
      1.0,      // 3: Flange width
      20,       // 4: Flange angle
      0.5,      // 5: Root radius
      [         // 6: Flange tip radii
        0.5,      // 
      ],
      0.2,      // 7: Chamfer depth  // Does not seem to be specified
      30,       // 8: Chamfer angle
      42,       // 9: Back to back measurement
      45,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "g1ScaleOne32",       // Not supported yet
    [
      4.5,      // 0: Tyre width
      3,        // 1: Tyre coning angle
      1.0,      // 2: Flange depth
      1.0,      // 3: Flange width
      0,        // 4: Flange angle  // Not enough room for this?
      0.5,      // 5: Root radius
      [         // 6: Flange tip radii
        0.5,      // 
      ],
      0.2,      // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      42,       // 9: Back to back measurement
      45,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "g11927ProfileA",       // Not currently supported...
    [
      0.175*smm,// 0: Tyre width
      3,        // 1: Tyre coning angle
      0035*smm, // 2: Flange depth
      0.036*smm,// 3: Flange width
      -1,       // 4: No angle but used different radii
      0.02*smm, // 5: Root radius
      [         // 6: Flange tip radius
        0.015*smm,
        0.020*smm,
        0.059*smm,
      ],
      0.008*smm,// 7: Chamfer depth
      45,       // 8: Chamfer angle
      42,       // 9: Back to back measurement
      45,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "OStandard",
    [
      4.4,      // 0: Tyre width
      3,        // 1: Tyre coning angle
      1.4,      // 2: Flange depth
      2.25/2,   // 3: Flange width Min 
      0,        // 4: Flange angle  // Not enough room for this?
      0.6,      // 5: Root radius
      [         // 6: Flange tip radii
        0.31,   // min 0.25, max 0.37
      ],
      0.2,      // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      28,       // 9: Back to back measurement
      32,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "OFine",
    [
      3.5,      // 0: Tyre width
      3,        // 1: Tyre coning angle
      1.1,      // 2: Flange depth
      1.75/2,   // 3: Flange width
      0,        // 4: Flange angle  // Not enough room for this?
      0.6,      // 5: Root radius
      [         // 6: Flange tip radii
        0.31,   // min 0.25, max 0.37
      ],
      0.2,      // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      29,       // 9: Back to back measurement
      32,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "OFine7",      // Not supported yet
    [
      3.5,      // 0: Tyre width
      3,        // 1: Tyre coning angle
      1.1,      // 2: Flange depth
      1.75/2,   // 3: Flange width
      0,        // 4: Flange angle  // Not enough room for this?
      0.6,      // 5: Root radius
      [         // 6: Flange tip radii
        0.31,   // min 0.25, max 0.37
      ],
      0.2,      // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      29,       // 9: Back to back measurement
      32,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "OO",
    [
      2.3,      // 0: Tyre width
      3,        // 1: Tyre coning angle
      0.51,     // 2: Flange depth
      0.8,      // 3: Flange width
      20,       // 4: Flange angle
      0.35,     // 5: Root radius (Guess)
      [         // 6: Flange tip radii
        0.35,
      ],
      0.25,     // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      14.4,     // 9: Back to back measurement (minimum)
      16.5,     // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "g3",    // 2.5in gauge See: https://www.gauge3.org.uk/technical-standards
             //  and http://cheltsme.org.uk/index-wheel_standards.html
    [
      0.268*smm,// 0: Tyre width
      3,        // 1: Tyre coning angle
      0.085*smm,// 2: Flange depth 0.080 sprung, 0.095 unsprung.
      0.063*smm,// 3: Flange width (thickness) - 0.055 cheltsme.org.uk (see below)
      10,       // 4: Flange angle each side. - 20deg cheltsme.org.uk
      0.035*smm,// 5: Root radius
      [         // 6: Flange tip radii
        0.02*smm,
      ],
      0.015*smm,// 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      2.281*smm,// 9: Back to back measurement
      64,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "3.5in",    // See: http://cheltsme.org.uk/index-wheel_standards.html
    [
      0.375*smm,// 0: Tyre width
      3,        // 1: Tyre coning angle
      0.11*smm, // 2: Flange depth
      0.075*smm,// 3: Flange width
      20,       // 4: Flange angle
      0.05*smm, // 5: Root radius
      [         // 6: Flange tip radii
        0.03*smm,
      ],
      0.02*smm, // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      3.281*smm,// 9: Back to back measurement
      89,       // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "5in",
    [
      0.535*smm,// 0: Tyre width
      3,        // 1: Tyre coning angle
      0.14*smm, // 2: Flange depth
      0.106*smm,// 3: Flange width
      20,       // 4: Flange angle
      0.07*smm, // 5: Root radius
      [         // 6: Flange tip radii
        0.045*smm,
      ],
      0.03*smm, // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      4.687*smm,// 9: Back to back measurement
      127,      // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "7.25in",
    [
      0.813*smm,// 0: Tyre width
      3,        // 1: Tyre coning angle
      0.183*smm,// 2: Flange depth
      0.14*smm, // 3: Flange width
      20,       // 4: Flange angle
      0.125*smm,// 5: Root radius
      [         // 6: Flange tip radii
        0.063*smm,
      ],
      0.04*smm, // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      6.813*smm,// 9: Back to back measurement
      184,      // 10: Gauge (Between inside of rails)
    ],
  ],
  [
    "default",    // NEEDS WORK IF THIS IS TO WORK (Only gauge currently calculated)
    [
      0.813*smm,// 0: Tyre width
      3,        // 1: Tyre coning angle
      0.183*smm,// 2: Flange depth
      0.14*smm, // 3: Flange width
      20,       // 4: Flange angle
      0.125*smm,// 5: Root radius
      [         // 6: Flange tip radii
        0.063*smm,
      ],
      0.04*smm, // 7: Chamfer depth  // Does not seem to be specified
      45,       // 8: Chamfer angle
      6.813*smm,// 9: Back to back measurement
      ((4*12)+8.5)*sf,  // 10: Gauge (Between inside of rails)
    ],
  ],
];

function wheelRimDepth(type)=
     type=="nHole"?3.0*sf
    :type=="mansell"?2.5*sf
    :type=="chaldron"?1.0*sf
    :type=="rchSpoked"?3.5*sf
    :type=="mk1Coach"?4.5*sf
    :type=="rchHole"?3.25/2*sf
    :2.5*sf;
function wheelProfile(standard)=let(cid=search([standard],wheelProfileData)) wheelProfileData[cid[0]][1];
function wheelStandard()=sf==Ogsf?
    wheelStandard=="standard"?"OStandard":"OFine"
  :
  (sf==g132sf || sf==g110mmsf || sf==1)? 
   wheelStandard=="standard"?"g1Standard":"g1Fine"
  :
    scale;

/**
  * railWheel(standard, type, config)
  * Builds a complete railway wheel.
  * Parameters:
  *   standard- string. The wheel rim standard 
  *   type    - string. The type of wheel
  *   dia     - mms. Wheel diameter
  *   axleDia - mms. The diameter of the axle (Hole in middle)
  *   hubDia  - mm. Centre hub diameter: 
  *   hubDelta- mm. default 0. Additional width of centre hub. Added to tyre width
  *   spokeConfig: array
  *     0 : Number of spokes
  *     1 : Type of spoke ("round", "split" or any other string for rectangular)
  *
  */    
module railWheel(
  standard="g1Standard",
  type="nHole",
  dia=((3*12)+1.0)*sf,
  axleDia=3.0,
  hubDia=0,
  hubDelta=0,
  spokeConfig=[],
  holeConfig=[]) {
  
  rd=wheelRimDepth(type);     // Depth - Size along wheel radii of rim.
  hd=dia-rd*2+_dx;    // Diameter to inside of rim
  
  cid=search([standard],wheelProfileData);
  wheelData=wheelProfileData[cid[0]][1];
  tw=wheelData[0];   // Tyre width
  hubWidth=tw+(hubDelta*sf);
  hubData=[hd,wheelData[0],hubDia,hubWidth,axleDia];

  difference() {
    union() {
      railWheelRim(wheelData,rd=rd,wd=dia);    // The basic wheel tread
      if (type=="nHole") {
        nHoleWheelHub(hubData,holeConfig);
      } else if (type=="mansell") {
        mansellWheelHub(hubData,true);
      } else if (type=="solid") {
        solidWheelHub(hubData);
      } else if (type=="rchHole") {
        rchDiskWheelHub(hubData,holeConfig);
      } else if (type=="chaldron") {
        chaldronWheelHub(hubData,spokeConfig);
      } else if (type=="spoked" || type=="rchSpoked") {
        spokedWheelHub(hubData,spokeConfig);
      } else if (type=="mk1Coach") {
        mk1CoachWheel(hubData);
      }
    }
    // Any parts to be removed for particular wheel types?
    if (type=="rchSpoked") {
      rchWheelRimDiff(dia,tw,rd);
    } else if (type=="mk1Coach") {
      mk1WheelRimDiff(dia,tw,rd);
    }
    translate([0,0,-_dx])
      cylinder(h=hubWidth*1.1,d=axleDia,center=false);
  }
}

module railWheelRim(config,rd,wd=37.5*sf) {
  d=wd/2;
  tw=config[0]; // Tyre width
  ca=config[1]; // Cone angle
  fd=config[2]; // Flange depth
  fw=config[3]; // Flange width
  fa=config[4]; // Flange angle
  rr=config[5]; // Root radius
  fr1=config[6][0];
  fr2=is_undef(config[6][1])? fr1:config[6][1];
  chd=config[7];
  cha=config[8];
  
  da=180/$fn;
  fa2=90-fa;
  
  // End points of the first curve
  xr1=(d+fd)-fr2*(1-cos(fa2));
  yr1=fr1+fr2*sin(fa2);
  
  // Start pont for the 2nd curve
  xr2=d+(rr*(1-sin(fa)));
  dx2=xr1-xr2; 
  
  // Check the values make sense
  assert(dx2>0,"bulbSection: - width is too narrow for specified radius r");
  // End point of the roor radius curve
  yr2=yr1+(dx2*tan(fa));
  yr3=yr2+rr*(sin(fa2)-sin(ca));

  dcx=(tw-yr3-chd)*sin(ca);  
  chx=chd/tan(cha);
  
  points=[
    [d-rd,0],
    [d,0],
    for (a=[0:da:90]) [(d+fd-fr1)+fr1*(sin(a)),fr1*(1-cos(a))],
    for (a=[0:da:fa2]) [(d+fd)-fr2*(1-cos(a)),fr1+fr2*(sin(a))],
    [xr1,yr1],
    [xr2,yr2],
    for (a=[fa2:-da:ca]) [d+rr*(1-cos(a)),yr2+rr*(sin(fa2)-sin(a))],
    [d,yr3],
    [d-dcx,tw-chd],
    [d-dcx-chx,tw],
    [d-rd,tw],
  ];
    
  // Improve the "roundness" of the rim if high fidelity output required
  rotate_extrude($fn=$fn<=30?$fn*2:$fn)
    polygon(points);  
}

/**
  * ***WheekHub() Draw Differens wheel hubs from the centre to the wheel rim
  * Builds a complete railway wheel.
  * Parameters:
  *   config:: array
  *     0  : diameter (dai) - mms. Diameter of whole hub (Inside rim)
  *     1  : rim width - mm. The width of the wheel rim defind by standard
  *     2  : hubDia - mm. Centre hub diameter: 
  *     3  : hubDelta - mm. default 0. Additional width of centre hub. Added to tyre width
  *     4  : axle diameter - mmd. The diameter of the axle (Hole in middle
  *
 */    
module solidWheelHub(config=[36.0*sf,5.0]) {
  dia=config[0];
  rw=config[1];
  hubDia=is_undef(config[2])?10.0*sf:config[2];
  hubWidth=is_undef(config[3])?10.0*sf:config[3];
  
  union() {
    translate([0,0,rw/2]) {
      cylinder(h=1.5*sf,d=dia,center=true);
    }
    cylinder(h=hubWidth,d=hubDia);
  }
}

/* Simple 3 hole wheel centre
 * config[5] Optional - could be used to alter the size of the 3 holes
 */
module nHoleWheelHub(config=[36.0*sf,5.0],holeConfig=[]) {
  dia=config[0];
  rw=config[1];
  hubDia=is_undef(config[2])?10.0*sf:config[2];
  hubWidth=is_undef(config[3])?10.0*sf:config[3];
  numHoles=is_undef(holeConfig[0])?3:holeConfig[0];
  holeDia=is_undef(holeConfig[1])?6.0*sf:holeConfig[1]*sf;

  difference() {
    union() {
      translate([0,0,rw/2]) {
        cylinder(h=1.5*sf,d=dia,center=true);
      }
      cylinder(h=hubWidth,d=hubDia);

      // Add some support?
      for (h=[0:numHoles-1]) {
        rotate([0,0,(360/(numHoles*2))+360/numHoles*h])
          translate([dia/4,0,rw/4])
            cube([dia/2,2.0*sf,rw/2],center=true);
      }
    }
    for (h=[0:numHoles-1]) {
      rotate([0,0,360/numHoles*h])
        translate([((dia+hubDia)/4)*0.95,0,0])
          cylinder(h=rw*1.1,d=holeDia);
    }
  }
}
 
module spokedWheelHub(config=[36.0*sf,5.0],spokeConfig=[8,""])  {
  dia=config[0];
  rw=config[1];
  
  hubDia=is_undef(config[2])?10.0*sf:config[2];
  hubWidth=is_undef(config[3])?10.0*sf:config[3];
  numSpokes=is_undef(spokeConfig[0])? 8:spokeConfig[0];
  typeSpoke=is_undef(spokeConfig[1])? "":spokeConfig[1];

  sa=360/numSpokes;   // Angle for each spoke
  
  union() {
    // Main hub secton
    roundedCylinder(h=hubWidth,d=hubDia,r=0.5*sf,center=false);
    translate([0,0,rw/2]) {
      for (h=[0:numSpokes-1]) {
        rotate([0,typeSpoke=="round"?90:0,sa*h]) {
          if (typeSpoke=="round") {
            cylinder(h=dia*0.5,r=0.75*sf);
            translate([0,0,dia*0.45])
              cylinder(h=2.0*sf,r1=0.75*sf,r2=1.5*sf);
          } else if (typeSpoke=="split") {
            rotate([0,0,-90])
              translate([-hubDia/2+(0.0*sf),0,0])
                splitSpoke((dia-hubDia)*0.5,numSpokes,hubDia);
            // Fillets at rim
            translate([-0.7,dia*0.5-0.5*sf,0])
              fillet(size=[(3+3/8)*sf,1*sf,1*sf]);
            translate([0,dia*0.5,0])    // Fills in some gaps
              cube([1.0*sf,1.0*sf,(3+3/8)*sf],center=true);
            mirror([1,0,0])
              translate([-0.7,dia*0.5-0.5*sf,0])
                fillet(size=[(3+3/8)*sf,1*sf,1*sf]);
          } else {
            rotate([0,90,90])
              translate([0,0,hubDia/2-0.05*sf])
                rchSpoke((dia-hubDia)*0.5+(0.2*sf));
            // Fillets at rim
            translate([-1.04*sf,dia*0.5-0.5*sf,0])
              fillet(size=[2.475*sf,1*sf,1*sf]);
            mirror([1,0,0])
              translate([-1.04*sf,dia*0.5-0.5*sf,0])
                fillet(size=[2.45*sf,1*sf,1*sf]);
            // Fillets at hub
            translate([-1.08*sf,hubDia/2+0.24*sf,0])
              rotate([0,0,-90])
                fillet(size=[2.875*sf,1*sf,1*sf]);
            translate([1.08*sf,hubDia/2+0.24*sf,0])
              rotate([0,0,180])
                fillet(size=[2.875*sf,1*sf,1*sf]);
          }
        }
      }
    }
  }
}

module rchWheelRimDiff(dia,tw,rd) {
  // RCH spoked wheel has deeper wheel rim which we then remove parts of to leave centre section
  difference() {
    cylinder(h=3.5*sf,d=dia-rd*2+0.75*sf,center=true);
    cylinder(h=3.6*sf,d=dia-rd*2-0.75*sf,center=true);
  }
  translate([0,0,tw]) {
    difference() {
      cylinder(h=3.5*sf,d=dia-rd*2+0.75*sf,center=true);
      cylinder(h=3.6*sf,d=dia-rd*2-0.75*sf,center=true);
    }
  }
}

module chaldronWheelHub(config,spokeConfig)  {
  dia=config[0];
  rw=config[1];
  
  hubDia=is_undef(config[2])?10.0*sf:config[2];
  hubWidth=is_undef(config[3])?10.0*sf:config[3];
  numSpokes=is_undef(spokeConfig[0])? 8:spokeConfig[0];
  typeSpoke=is_undef(spokeConfig[1])? "":spokeConfig[1];
  spokeDia=is_undef(spokeConfig[2])? 1.5:spokeConfig[2];

  sa=360/numSpokes;   // Angle for each spoke
  
  union() {
    // Main hub secton
    roundedCylinder(h=hubWidth,d=hubDia,r=0.5*sf,center=false);
    // Flange at back of wheel
    roundedCylinder(h=1.5*sf  ,r=0.5*sf,d=hubDia*(typeSpoke=="round"?1.25:1.5),center=false);
    translate([0,0,hubWidth*0.48]) {
      for (h=[0:numSpokes-1]) {
        rotate([0,typeSpoke=="round"?90:0,sa*h]) {
          if (typeSpoke=="round") {
            cylinder(h=dia*0.5,d=spokeDia*sf);
            translate([0,0,dia*0.45])
              cylinder(h=2.0*sf,d1=spokeDia*sf,d2=spokeDia*2*sf);
          } else if (typeSpoke=="split") {
            rotate([0,0,180])
              translate([-hubDia/2,0,0])
                splitSpoke(l=(dia-hubDia)*0.5,numSpokes,hubDia,spokeWidth=0.75*sf);
            // Fillet at rim
              rotate([0,90,0])
                translate([-0.14*sf,0,dia*0.5-2.0*sf])
                  scale([1.8,1,1])
                    cylinder(h=2.0*sf,r1=0.5*sf,r2=1.75*sf);
          } else {
            translate([0,0,1.5*sf])
              rotate([0,90,0])
                extrudedFlatEgg(r1=0.5*sf,r2=0.5*sf,3.0*sf,dia*0.5,center=false);
            // Fillet at rim
            rotate([0,90,0])
              translate([-0.14*sf,0,dia*0.5-2.0*sf])
                scale([1.8,1,1])
                  cylinder(h=2.0*sf,r1=0.5*sf,r2=1.75*sf);
          }
        }
      }
    }
  }
}

module mansellWheelHub(config,showSegments=true) {
  dia=config[0];
  rw=config[1];
  hubDia=is_undef(config[2])?10.0*sf:config[2];
  hubWidth=is_undef(config[3])?10.0*sf:config[3];
  outerRim(dia,rw);
  
  iri=0.75*sf;      // Inner rim inset from 0 3/8 * 2 (wooden part of wheel)
  irt=rw-iri*2;     // Depth (thickness of inner rim);

  // The hub
  ih1d=21*sf;       // Diameter to center metal flange
  ih1t=irt+(5/8+7/8*2)*sf;
  
  x=360/16;     // Angle of each segment
  // Segmeted wooden section of wheel
  difference() {
    translate([0,0,iri]) {
      cylinder(h=irt,d=dia);
    }
    // Show the individual segments?
    if (showSegments) {
      for (s=[0:31]) {
        rotate([0,0,s*x]) {
          translate([hubDia,0,rw-1*sf])
            cube([dia/2,0.4*sf,1*sf],center=false);
          translate([hubDia,0,0.5*sf])
            cube([dia,0.4*sf,1*sf],center=false);
        }
      }
    }
  }

  // The hub of the wheel
  ha=hubWidth-rw;   // Any additional width to outside of inner hub
  translate([0,0,-5/8*sf])
    hub(ih1d,hubDia,ih1t,ha=ha);
  
  a=360/32;     // Angle of each segment
  for (s=[0:31]) {
    rotate([0,0,s*a]) {
      // Outer ring of rivets
      if (s%2) {
        rotate([0,0,a/4]) {
          translate([dia/2-1.75*sf,0,rw-3/8*sf])
            screwHead(h=0.5*sf,d=1.25*sf,type="rivet");
          translate([dia/2-1.75*sf,0,3/8*sf+_dx])   // 3/8 is depth of inner part of outer rim from 0
            mirror([0,0,1])
              screwHead(h=0.8*sf,d=1.25*sf,type="nut");
        }
      } else {
        translate([ih1d/2-((s%4)?1.8*sf:4.0*sf),0,rw])
          screwHead(h=0.5*sf,d=1.25*sf,type="rivet");
        translate([ih1d/2-((s%4)?1.8*sf:4.0*sf),0,0.25*sf/2])
          mirror([0,0,1])
            screwHead(h=0.8*sf,d=1.25*sf,type="nut");
      }
    }
  }

  module outerRim(dia,ww) {
    r=dia/2;
    rd=3.0*sf;
    rd1=0.5*sf;
    d1=3/8*sf;
    da=180/($fn>20?20:$fn);
    
    points=[
     [r,0],
     [r,ww],
     [r-rd1,ww],
     for (a=[0:da:90]) [r-rd1-d1*(1-cos(a)),ww-d1*sin(a)],
     [r-rd1-d1,ww-d1],
     [r-rd+d1,ww-d1],
     for (a=[0:da:90]) [r-rd+d1*(1-sin(a)),ww-d1*(2-cos(a))],
     [r-rd,ww-d1*2],
     [r-rd,d1*2],
     for (a=[90:-da:0]) [r-rd+d1*(1-sin(a)),d1*(2-cos(a))],
     [r-rd+d1,d1],
     [r-rd1-d1,d1],
     for (a=[90:-da:0]) [r-rd1-d1*(1-cos(a)),d1*sin(a)],
     [r-rd1,0],
    ];
    
    rotate_extrude(angle=360)
      polygon(points);
  }
  
  module hub(dia,dia1,ww,ha=0) {
    r=dia/2;
    r1=dia1/2;
    // The following are guesses to be confirmed
    t1=0.25*sf;     // Outside wheel inner hub depth
    t2=0.75*sf;     // Inside wheel inner hub depth
    r4=1.5*sf;
    ax=60;          // Approximate only
    rd=3.0*sf;
    rd1=0.5*sf;
//    d1=3/8*sf;
//    d2=5/8*sf;
    d1=1.0*sf;
    d2=1.0*sf;   
    da=180/($fn>20?20:$fn);
    
    points=[
     [r-d2,t2],
     for (a=[90:-da:0]) [r-d2*(1-cos(a)),t2+d2*(1-sin(a))],
     [r,d2+t2],
     [r,ww-t1-d1],
     for (a=[0:da:90]) [r-d1*(1-cos(a)),ww-t1-d1*(1-sin(a))],     
     [r-d1,ww-t1],     
     [r1,ww-t1],     
     [r1,ww+ha],
     [0,ww+ha],
     [0,0],
     [r1,0],
     for (a=[ax:-da:0]) [r1+r4*(0.9-sin(a)),-t2+r4*(cos(a))],     
     [r1+r4*0.9,t2],
    ];
    
    rotate_extrude(angle=360)
      polygon(points);
  }
}

module mk1CoachWheel(config,sups=false) {
  dia=config[0];
  tw=config[1];
  hubDia=is_undef(config[2])?10.0*sf:config[2];
  hubWidth=is_undef(config[3])?10.0*sf:config[3];

  hr=1.75*sf;    // Radius of hub to outside wheel
  hd=tw-hr*2;    // Hub depth (thickness)

  translate([0,0,tw/2]) 
  {
    difference() {
      union() {
        wheelHub(dia=dia+0.5*sf,ww=tw,hw=hubWidth);
      }
      for (h=[0:3]) {
        rotate([0,0,90*h])
          translate([12.5*sf,0,0])
            cylinder(h=10*sf,d=2.0*sf,center=true);
      }
    }
  }
  if (sups) {
    n=3;
    dx=dia/(n+2);
    for (c=[0:1:n-1]) {
      difference() {
        cylinder(h=1.54*sf-0.1,d=8.0*sf+(dx*(c+1)));
        translate([0,0,-0.01])
          cylinder(h=2.0*sf,d=8.0*sf+(dx*(c+1))-0.4);
          
        for (r=[0:1:3]) {
          rotate([0,0,r*45])
            cube([dia,4.5*sf,4.0*sf],center=true);
        }
//        cube([5.0*sf,dia,4.0*sf],center=true);
      }
    }
  }

  module wheelHub(dia,ww,hw) {
    union() {
      cylinder(h=ww,d=hubDia,center=true);
      translate([0,0,ww/2-hr-hd])
        cylinder(h=hd,d=dia);
      translate([0,0,ww/2-hr]) {
        elipticHorn(h=hr+hw-ww,d1=hubDia*1.8,d2=hubDia);
        translate([0,0,-0.2*sf])
          rotate_extrude()
            translate([11.25*sf,0,0])
              oval(w=7*sf,h=1.0*sf,r=0.1*sf);
      }        
    }
  }
}

module mk1WheelRimDiff(dia,tw,rd) {
  hr=1.75*sf;    // Radius of hub to outside wheel
  dd=dia-rd*2-(1.75*sf*0.571);
  translate([0,0,tw])
    doughnut(d=dd,t=hr*2);
  doughnut(d=dd,t=hr*2);
}

/** rchSpoke()
  * Used to create a tapered spoke similar to the RCH profile
  *   The profile is rectangular with rounded front and rear
  *   Sizes are based on the rch wheel drawings
  */
module rchSpoke(l=15*sf) {
  widr=2.75*sf;         // Width of spoke at rim
  sthe=(1+1/16)*sf;     // Spoke thickness at rim
  re=(1+1/8)*sf;
  widh=3.25*sf;         //  Width of spoke at hub
  sth=(1+3/16)*sf;      // Spke thickness at hub
  
  sx=widr/widh;         // Scale X for tappered spoke
  sy=sthe/sth;          // Scale Y
  cx=widh/2-re;         // Centre of side radii
  a=asin((sth/2)/re);   // Angle of side radii
  px=cx+re*cos(a);      // Position of centre of radii
  da=180/($fn<12?12:$fn);

  points=[
    [px,-sth/2],
    for (b=[-a:da:a]) [cx+re*cos(b),re*sin(b)],
    [px,sth/2],
    [-px,sth/2],
    for (b=[180-a:da:180+a]) [-cx+re*cos(b),re*sin(b)],
    [-px,-sth/2],
  ];
  
  linear_extrude(l,scale=[sx,sy],center=false)
    polygon(points);
}

module rchDiskWheelHub(config,holeConfig=[]) {
  dia=config[0];
  rw=config[1];
  hubDia=is_undef(config[2])?8.5*sf:config[2];
  numHoles=is_undef(holeConfig[0])?3:holeConfig[0];
  holeDia=is_undef(holeConfig[1])?6.0*sf:holeConfig[1]*sf;

  difference() {
    rotate_extrude(angle=360)
      rchDiskWheelProfile(config);
    for (h=[0:numHoles-1]) {
      rotate([0,0,(360/numHoles)*h])
        translate([((dia+hubDia)/4)*0.97,0,0])
//        translate([is_undef(config[6])?dia*0.32:config[6],0,0])
          cylinder(h=rw*1.1,d=holeDia);
    }
  }
}

module rchDiskWheelProfile(config) {
 /* The wheel hub and disk diameter should be 2ft 10.25" for 3ft 1.5" wheel (inside dia of rim)
  * which implies wheel rim depth should be 3.25/2" = 1.625"
  * The first radius centre of 1.5" is 2.25" in below this (0.5" from center) as we
  * assume thickness of profile ia always the same although on at rim actually 3/4" and 1" at hub
  * we will make it it the same all the way through
  */
  dia=config[0];
  rw=config[1];
  hubDia=is_undef(config[2])?10.0*sf:config[2];
  hubWidth=is_undef(config[3])?10.0*sf:config[3];

  t=1.0*sf;     // disk profile thickness
  rcl=rw/2;     // Centre line of disk at rim

  rtO=(rw+t)/2; // Offset from center at outside rim
  rtI=(rw-t)/2; // Offset from center at inside rim
  
  r=dia/2;
  da=180/($fn<20?20:$fn);
  
  // Naming: xxxO == Outside of wheel, xxxI = inside (wheel flange side) xxxB = both
  rrB=1.5*sf;       // Radius at rim of wheel between rim and disk
  rrcxB=2.25*sf;    // X Center of radius rrB from rim
  rrcyI=rtI-rrB;    // Y Center of radius rrB from 0 at inside of rim
  rcxB=r-rrcxB;     // X center of radius rrB for x=0. (Hub centre)
  
  // Now calculate the end angle of Inside curve
  mI=sqrt(rrcxB^2 + rrcyI^2);
  // sin(u1) = rrB/mI
  u1=asin(rrB/mI);
  // sin(u2)= rrcyI/mm
  u2=asin(rrcyI/mI);
  arI = u1+u2;
  
  // There is a straight bit on the outside 2.125" from wheel dia
  rocxO=(2.125-1.625)*sf;
  m2=sqrt((rrcxB-rocxO)^2 + rrcyI^2);
  u3=asin(rrB/m2);
  u4=asin(rrcyI/m2);
  arO = u3+u4;

  // Now calculate the hub locations
  hcl=(2.25-0.75)*sf;     // Center line from base of profile at hub (We are cutting off 3/4" of the hub
  // hrsB=0.5*sf;            // edge radius at hub not used currently
  ah=20;                  // Angle at hub
  hrI=2.0*sf;             // Radius at hub - Should be 2.0" but breaks the code as hub not full length
  hrO=2.0*sf;             // Radius at hub
   
  // Inside and outside (flange side) will have an offset caused by the truncated hub
  hdxB=0.75*sf*sin(ah);    // Approx due to cut of 0.5" radius
  hbB=hubDia/2+hdxB;       // X position of bass edge
  hrcyO=hcl+t/2+hrO;       // Y offset of centre of outside hub radius
  yO=rw-hrcyO;             // Y offset from radius centre to edge of hub
  hrcxO=hbB+yO*tan(ah)+hrO;
  //  Calculate centre of offside hub radius centre
  // sin(20) = hoyI/hrB;
  hrcyI=hcl-t/2-hrI;     // Y offset of centre of inside hub radius
  
  yI=hrcyI;
  hrcxI=hbB+yI*tan(ah)+hrI;
  
  // Calculations for S curve in centre of wheel
  // Cord width
  hI = (rcl-hcl)/2;
  hO = (rcl-hcl)/2;
  hrcxB=max(hrcxO,hrcxI);  // Use max X of hub bens for S bend location  as inside radius is odd.
  cO = (rcxB-hrcxB)/2;
  cI = (rcxB-hrcxB)/2;
  // L = sqrt(c1^2+h^2)
  // Using similar triangles h/L = L/2r => 2rh = L^2 => r = L^2/2
  rO = (cO^2 + hO^2)/(2*hO);
  rI = (cI^2 + hI^2)/(2*hI);
  // sin(a1) = c1/r1; =>
  aO=asin(cO/rO);
  aI=asin(cI/rI);
  
  points=[
   // Inside of hub
   [0,0],
   [hubDia/2,0],
   // Inside hub radius
   [hrcxI-hrI*cos(ah),max(hrcyI+hrI*sin(ah),0)],
   for (a=[ah:da:90]) [hrcxI-hrI*(cos(a)),max(hrcyI+hrI*(sin(a)),0)],
   [hrcxI,hrcyI+hrI],
   [hrcxB,hrcyI+hrI],   // Staight bit up to S bend
   
   // Inside S bend
   for (a=[0:da:aI]) [hrcxB+rI*sin(a),hrcyI+hrI+rI*(1-cos(a))],
   [hrcxB+rI*sin(aI),hrcyI+hrI+rI*(1-cos(aI))],
   for (a=[aI:-da:0]) [rcxB-rI*sin(a),rtI-rI*(1-cos(a))],
   [rcxB,rtI],
   for (a=[0:da:arI]) [rcxB+rrB*sin(a),rrcyI+rrB*cos(a)],
   [rcxB+rrB*sin(arI),rrcyI+rrB*cos(arI)],
   [r,0],
   
   // Switch to outside of wheel at rim #O
   [r,rw],
   [r-rocxO,rw],
   [rcxB+rrB*sin(arO),rw-rrcyI-rrB*cos(arO)],
   for (a=[arO:-da:0]) [rcxB+rrB*sin(a),rw-rrcyI-rrB*cos(a)],
   [rcxB,rtO],
   // Outside S bend here
   for (a=[0:da:aO]) [rcxB-rO*sin(a),rtO-rO*(1-cos(a))],
   [rcxB-rO*sin(aO),rtO-rO*(1-cos(aO))],
   for (a=[aO:-da:0]) [hrcxO+rO*sin(a),hcl+t/2+rO*(1-cos(a))],
   [hrcxO,hcl+t/2],
   
   // Outside hub radius
   for (a=[0:da:90-ah]) [hrcxO-hrO*sin(a),hcl+t/2+hrO*(1-cos(a))],    
   [hrcxO-hrO*sin(90-ah),hcl+t/2+hrO*(1-cos(90-ah))],
   [hbB,rw],
   [hubDia/2,hubWidth],
   [0,hubWidth],
   [0,rw],
   
  ];
  polygon(points);  
}

/** rchSpoke()
  * Used to create a split spoke similar to the RCH profile rch1024.pdf
  * Parameters:
  *   l  : Length of spoke (hub to rim)
  *   n  : Number of spokes needed for correct angles
  */
module splitSpoke(l=8*sf,n=8,hd=10*sf,spokeWidth=0) {
  wid=(3+3/8)*sf;         // Width of spoke

//  sthf=(3/8)*sf;          // Spoke thickness at front (not used)
//  sthr=(5/8)*sf;          // Spoke thickness at rear as per RCH spec
  sthr=(spokeWidth==0)?l*0.07:spokeWidth;              // Proportional to wheel size
  
  ah=360/(n*3);           // Angle of each segment at hub
  ah2=ah*2;
  
  dy=hd*sin(ah);          // Location of each split spoke on hub for centre
  ro=(dy+sthr);           // No specification for radius for hub end
  ri=(dy-sthr);
  cyo=-ro/2;
  cyi=-ri/2;
  
  // Spoke needs to be longer
  len=l+hd*(1-cos(ah));
  cxo=len-(ro/2*sin(ah));
  cxi=len-(ri/2*sin(ah));
  aro=asin(ro/2/cxo);
  ari=asin(ri/2/cxi);
  da=ah/20;
  sy=sthr/sthr;
//  translate([cxo+ro*sin(-ah),-ro*cos(-ah)]) color("red") circle(0.1); // Debug positions
  
  points=[
    [0,sthr/2],
    [cxo+ro*sin(-aro),cyo+ro*cos(-aro)],
    for (a=[-aro:da:ah2]) [cxo+ro*sin(a),cyo+ro*cos(a)],
    [cxo+ro*sin(ah2),cyo+ro*cos(ah2)],
    [cxi+ri*sin(ah2),cyi+ri*cos(ah2)],
    for (a=[ah2:-da:-ari]) [cxi+ri*sin(a),cyi+ri*cos(a)],
    [cxi+ri*sin(-ari),cyi+ri*cos(-ari)],
    [sthr*2,0],       // Mathmatically speaking for inner tangents should be [0,0]
    [cxi+ri*sin(-ari),-cyi-ri*cos(-ari)],
    for (a=[-ari:da:ah2]) [cxi+ri*sin(a),-cyi-ri*cos(a)],
    [cxi+ri*sin(ah2),-cyi-ri*cos(ah2)],
    [cxo+ro*sin(ah2),-cyo-ro*cos(ah2)],
    for (a=[ah2:-da:-aro]) [cxo+ro*sin(a),-cyo-ro*cos(a)],
    [cxo+ro*sin(-aro),-cyo-ro*cos(-aro)],
    [0,-sthr/2],  
  ];
  
  translate([-len,0,0])
    linear_extrude(wid,center=true)
      polygon(points);
}
